`timescale 1ns/1ps

module testbench;
//-----------------------------------------------------------------------------
// Enable verification summary for console
// Warning! Verification may slow down simulation.
`define __VER
//-----------------------------------------------------------------------------

//========================================================================
// Array constants table  
//========================================================================
parameter numWLP  = 8;
parameter numWL   = 1024;
parameter numMux  = 256;
parameter numSect = 8;

parameter D_MSB = 8;
parameter A_MSB = 21;
parameter A_SEC_MSB = 18;

parameter sizePage   = numWLP * numMux;
parameter sizeSector = numWL * numMux;
parameter sizeMem    = numSect * sizeSector;
//========================================================================
// Timing units table  
//========================================================================
parameter ns_unit  = 1;
parameter us_unit  = 1000;
parameter ms_unit  = 1000 * 1000;
parameter sec_unit = 1000 * 1000 * 1000;
//========================================================================
// Timing table for PARALLEL I/F  
//========================================================================
parameter	Tvcs = 100_000;	// Ucc setup time in ns

parameter Ta_noe = 56;
parameter Ta_nce = 66;
parameter Ta_a = 56;

parameter Tsu_a_wel = 0;
parameter Th_a_wel = 45;
parameter Tsu_d_weh = 30;
parameter Th_d_weh = 15;

parameter Tsu_a_cel = 0;
parameter Th_a_cel = 45;
parameter Tsu_d_ceh = 30;
parameter Th_d_ceh = 15;

parameter Tsu_cel_wel = 0;
parameter Tsu_oeh_wel = 0;
parameter Th_weh_cel  = 0;

parameter Tsu_wel_cel = 0;
parameter Tsu_oeh_cel = 0;
parameter Th_ceh_wel  = 0;

parameter Th_oeh_weh = 13;

parameter Twpl = 30;
parameter Twph = 35;
parameter Twph_half = 17.5;

parameter Tcpl = 30;
parameter Tcph = 35;
parameter Tcph_half = 17.5;

parameter Tcyr = 70;
parameter Tcyw = 70;

parameter Tpdz_oe_d = 16 * 2; 

parameter Tw_sec_load = 49; // Timeout window for sector load in us

parameter Tw_we_pr = 100; // in us
parameter Tw_pr_we = 40; // in ms
parameter Tsu_oe_pr = 4; // in us
parameter Tsu_a9_pr = 4; // in us

integer i, j;

//------------------------------------------------------------------------
// I/O Ports                                                           
//------------------------------------------------------------------------
reg  [A_MSB-1:0]  A; 
wire [D_MSB-1:0]  D;
reg  [D_MSB-1:0]  DIN_FLASH;
reg  [D_MSB-1:0]  DOUT_FLASH;

reg               nCE;
reg               nWE;
reg               nOE;

reg               A9_HV;
reg               OE_HV;

//------------------------------------------------------------------------
// Testbench routines                                                   
//------------------------------------------------------------------------
reg  [D_MSB-1:0]  rdbuf;  // read  buffer
reg  [D_MSB-1:0]  rdbufm; // read  buffer
reg  [D_MSB-1:0]  wrbuf;	// write buffer

reg               ce_mod;
reg               drive;
reg  [A_MSB-1:0]  ap;
reg               match;
reg               dismatch;

integer test_no;

reg [D_MSB-1:0] mem [sizeMem-1:0];

assign D[7:0]  = !nOE && !nCE && nWE ? 8'hz : drive ? DIN_FLASH : 8'hz;

//=============================================================================
//===== TOP Device =====
//=============================================================================
K1636RR4U dut (
  .nWE(nWE),
  .nOE(nOE),
  .nCE(nCE),
  .A(A),
  .D(D),
  .A9_HV(A9_HV),
  .OE_HV(OE_HV),
  .TDI(),
  .TCK(),
  .STROBE(),
  .MRST(),
  .SCK(),
  .SI(),
  .SEL_SPI(),
  .SO()
); 

//=============================================================================
//===== Initializing ports  =====
//=============================================================================
initial begin
  A = {A_MSB{1'bx}};
  DIN_FLASH = {D_MSB{1'bz}};

  nWE = 1'b1;
  nOE = 1'b1;
  nCE = 1'b1;

  A9_HV = 1'b0;
  OE_HV = 1'b0;

  ap = {A_MSB{1'b0}};
  match = 1'b1;
  dismatch = 1'b0;
  drive = 1'b1;
  ce_mod = 1'b0;
end


//=============================================================================
//===== Test sequence itself =====
//=============================================================================
initial begin
  #(Tvcs); // Wait for model wakeup

  $display("RUN[TEST]: Parallel I/F testing");

//=== Test1 - Autoselect ===========================================
  test_no=1;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  CMD_Autoselect; // Read ID
  CMD_Reset;
  for (i = 0; i < numSect; i = i + 1) begin
    CMD_Autoselect_RdPrSec(ap[A_MSB-1:A_SEC_MSB]); // Verify sector protect status
    CMD_Reset;
    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end

  ce_mod = 1'b1;

  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  CMD_Autoselect; // Read ID
  CMD_Reset;
  for (i = 0; i < numSect; i = i + 1) begin
    CMD_Autoselect_RdPrSec(ap[A_MSB-1:A_SEC_MSB]); // Verify sector protect status
    CMD_Reset;
    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end

//=== Test2 - Sector Protection ====================================
  test_no=2;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < numSect; i = i + 1) begin
    CMD_Protect_Sector(ap[A_MSB-1:A_SEC_MSB]); // Protect all sectors
    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end
//------------------------------------------------------------------
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < 3; i = i + 1) begin
    CMD_Unprotect_Sector(ap[A_MSB-1:A_SEC_MSB]); // Unprotect sectors from 1st to 3rd
    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end

  ap = {3'h7, {A_SEC_MSB{1'b0}}};
  CMD_Unprotect_Sector(ap[A_MSB-1:A_SEC_MSB]); // Unprotect sectors from 7th

//=== Test3 - Erase Chip ===========================================
  test_no=3; 
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  CMD_Erase_Chip;
//------------------------------------------------------------------
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (i = 0; i < 3; i = i + 1) begin // Read 1st page from 1st, 2nd, 3rd sectors
    CMD_Read_by_nOE(ap, sizePage); // Read by nOE
    #100;

    CMD_Read_by_nCE(ap, sizePage); // Read by nCE
    #100;

    CMD_Read_by_A(ap, sizePage); // Read by adress
    #100;

    ap[A_MSB-1:A_SEC_MSB] = ap[A_MSB-1:A_SEC_MSB] + 1;
  end

//=== Test4 - Program  ============================================
  test_no=4; 
  ap = {3'h0, 7'h00, 3'h0, 8'h0};

  for (i=0; i<2*sizePage; i=i+1) begin
    wrbuf = $random;
    CMD_Program(ap, wrbuf); // Program 2 first pages in 1st sector with random data
    ap = ap + 1;
  end

//----------------------------------------------------------------
  ap = {3'h0, 7'h00, 3'h0, 8'h0};

  CMD_Read_by_nOE(ap, 4*sizePage); // Read four pages by nOE
  #100;

  CMD_Read_by_nCE(ap, 4*sizePage); // Read four pages by nCE
  #100;

  CMD_Read_by_A(ap, 4*sizePage); // Read four pages by adress
  #100;

//----------------------------------------------------------------
  ap = {3'h7, 7'h7E, 3'h0, 8'h0};

  for (i=0; i<2*sizePage; i=i+1) begin
    wrbuf = $random;
    CMD_Program(ap, wrbuf); // Program 2 last pages in 8th sector with random data
    ap = ap + 1;
  end

//----------------------------------------------------------------
  ap = {3'h7, 7'h7C, 3'h0, 8'h0};

  CMD_Read_by_nOE(ap, 4*sizePage); // Read four pages by nOE
  #100;

  CMD_Read_by_nCE(ap, 4*sizePage); // Read four pages by nCE
  #100;

  CMD_Read_by_A(ap, 4*sizePage); // Read four pages by adress
  #100;

//=== Test5 - Page Erase =========================================
  test_no=5; 
  ap = {3'h0, 7'h00, 3'h0, 8'h0};

  CMD_Erase_Page(ap[A_MSB-1:A_SEC_MSB], ap[A_SEC_MSB-1:11]); // Erase 1st page from 1st sector

//----------------------------------------------------------------
  ap = {3'h0, 7'h00, 3'h0, 8'h0};

  CMD_Read_by_nOE(ap, 2*sizePage); // Read two first pages from 1st sector by nOE
  #100;

  CMD_Read_by_nCE(ap, 2*sizePage); // Read two first pages from 1st sector by nCE
  #100;

  CMD_Read_by_A(ap, 2*sizePage); // Read two first pages from 1st sector by adress
  #100;

//----------------------------------------------------------------
  ap = {3'h7, 7'h7F, 3'h0, 8'h0};

  CMD_Erase_Page(ap[A_MSB-1:A_SEC_MSB], ap[A_SEC_MSB-1:11]); // Erase 128th page from 8th sector

//----------------------------------------------------------------
  ap = {3'h7, 7'h7E, 3'h0, 8'h0};

  CMD_Read_by_nOE(ap, 2*sizePage); // Read two last pages from 16th sector by nOE
  #100;

  CMD_Read_by_nCE(ap, 2*sizePage); // Read two last pages from 16th sector by nCE
  #100;

  CMD_Read_by_A(ap, 2*sizePage); // Read two last pages from 16th sector by adress
  #100;

//=== Test6 - Sector Erase =========================================
  test_no=6; 
  ap = {3'h0, 7'h00, 3'h0, 8'h0};

  CMD_Erase_Sector(
		   0,   // 1st sector for erase 
		   7,   // 2nd sector for erase (optional)
		   0,   // 3rd sector for erase (optional)
		   0,   // 4th sector for erase (optional)
		   0,   // 5th sector for erase (optional)
		   0,   // 6th sector for erase (optional)
		   0,   // 7th sector for erase (optional)
		   0,   // 8th sector for erase (optional)
		   2);  // number of sectors for erase sector buffer (must be > 0)

//----------------------------------------------------------------
  ap = {3'h0, 7'h00, 3'h0, 8'h0};

  CMD_Read_by_nOE(ap, sizeSector);  // Read 1st sector by nOE
  #100;

  CMD_Read_by_nCE(ap, sizeSector); // Read 1st sector by nCE
  #100;

  CMD_Read_by_A(ap, sizeSector); // Read 1st sector by adress
  #100;
//----------------------------------------------------------------
  ap = {3'h7, 7'h00, 3'h0, 8'h0};

  CMD_Read_by_nOE(ap, sizeSector);  // Read 16th sector by nOE
  #100;

  CMD_Read_by_nCE(ap, sizeSector); // Read 16th sector by nCE
  #100;

  CMD_Read_by_A(ap, sizeSector); // Read 16th sector by adress
  #100;

//===================================================================
  `ifdef __VER
  if (dismatch) $display("ERROR[TEST]: Some operations unsuccessfully finished or read data is not matched with ethalon!");
  else $display("DONE[TEST]: All operations successfully finished and data is matched with ethalon");
  `else
  else $display("DONE[TEST]: All operations successfully finished");
  `endif

  $display("\n---< See ya later, allagator! >---\n");

  $finish;
end

always @ (negedge match) dismatch = 1; // dismatch test detector

//=============================================================================
//===== TASKS =====
//=============================================================================
task Seq;
input [11:0] addr;
input [7:0]  data;
input first;
input last;
begin
if (ce_mod) begin
  fork 
    if (first) #(Tcph_half-Tsu_wel_cel) nWE = 0;
    if (first) #(Tcph_half-Tsu_oeh_cel) nOE = 1;
    #(Tcph_half-Tsu_a_cel)              A[11:0] = addr;
    #(Tcph_half)                        nCE = 0;
    #(Tcph_half+Th_a_cel)               A = {A_MSB{1'bx}};
    #(Tcph_half+Tcpl)                   nCE = 1;
    if (last) #(Tcph_half+Tcpl+Th_ceh_wel) nWE = 1;
    #(Tcph_half+Tcpl-Tsu_d_ceh)         DIN_FLASH[7:0] = data;
    #(Tcph_half+Tcpl+Th_d_ceh)          DIN_FLASH = {D_MSB{1'bx}};
    #(Tcyw);
  join
end else begin
  fork 
    if (first) #(Twph_half-Tsu_cel_wel) nCE = 0;
    if (first) #(Twph_half-Tsu_oeh_wel) nOE = 1;
    #(Twph_half-Tsu_a_wel)              A[11:0] = addr;
    #(Twph_half)                        nWE = 0;
    #(Twph_half+Th_a_wel)               A = {A_MSB{1'bx}};
    #(Twph_half+Twpl)                   nWE = 1;
    if (last) #(Twph_half+Twpl+Th_weh_cel) nCE = 1;
    #(Twph_half+Twpl-Tsu_d_weh)         DIN_FLASH[7:0] = data;
    #(Twph_half+Twpl+Th_d_weh)          DIN_FLASH = {D_MSB{1'bx}};
    #(Tcyw);
  join
end
end
endtask

task Event;
input [A_MSB-1:0] addr;
input [D_MSB-1:0] data;
input first;
input last;
begin
if (ce_mod) begin
  fork 
    if (first) #(Tcph_half-Tsu_wel_cel) nWE = 0;
    if (first) #(Tcph_half-Tsu_oeh_cel) nOE = 1;
    #(Tcph_half-Tsu_a_cel)              A = addr;
    #(Tcph_half)                        nCE = 0;
    #(Tcph_half+Th_a_cel)               A = {A_MSB{1'bx}};
    #(Tcph_half+Tcpl)                   nCE = 1;
    if (last) #(Tcph_half+Tcpl+Th_ceh_wel) nWE = 1;
    #(Tcph_half+Tcpl-Tsu_d_ceh)         DIN_FLASH = data;
    #(Tcph_half+Tcpl+Th_d_ceh)          DIN_FLASH = {D_MSB{1'bx}};
    #(Tcyw);
  join
end else begin
  fork 
    if (first) #(Twph_half-Tsu_cel_wel) nCE = 0;
    if (first) #(Twph_half-Tsu_oeh_wel) nOE = 1;
    #(Twph_half-Tsu_a_wel)              A = addr;
    #(Twph_half)                        nWE = 0;
    #(Twph_half+Th_a_wel)               A = {A_MSB{1'bx}};
    #(Twph_half+Twpl)                   nWE = 1;
    if (last) #(Twph_half+Twpl+Th_weh_cel) nCE = 1;
    #(Twph_half+Twpl-Tsu_d_weh)         DIN_FLASH = data;
    #(Twph_half+Twpl+Th_d_weh)          DIN_FLASH = {D_MSB{1'bx}};
    #(Tcyw);
  join
end
end
endtask

task Save_Data;
input [A_MSB-1:0] ap;
begin
  `ifdef __VER
  match=1'b1;
  `endif

  DOUT_FLASH = D;

  `ifdef __VER
  rdbuf=DOUT_FLASH;
  rdbufm=mem[ap];
  if (rdbuf!=rdbufm) begin
    match=1'b0;
    $display("ERROR[SCH vs Ethalone]: rdbuf(%h)!=rdbufm(%h), addr=%h, DISMATCH DATA =( : time=",rdbuf,rdbufm, ap, $time);  
  end
  `endif  
end
endtask

task CMD_Read_by_nOE;
input [A_MSB-1:0] addr;     // initial address to read
input [31:0]      rdbytes;  // number of bytes to read

reg   [A_MSB-1:0] ap;
integer i;
begin
  match = 1'b1;
  ap = addr;
  drive = 1'b0;

  nCE = 0;
  nWE = 1;
  #100;
  for (i = 0; i < rdbytes; i = i+1) begin
    fork
      A = ap;
      #(Th_oeh_weh) nOE = 0;
      #(Th_oeh_weh+Ta_noe) Save_Data(ap);
      #(Tcyr) nOE = 1;
    join 
    ap = ap + 1;  
  end   
  nCE = 1;
  #(Tpdz_oe_d) drive = 1'b1;
end
endtask

task CMD_Read_by_nCE;
input [A_MSB-1:0] addr;     // initial address to read
input [31:0]      rdbytes;  // number of bytes to read

reg   [A_MSB-1:0] ap;
integer i;
begin
  match = 1'b1;
  ap = addr;
  drive = 1'b0;
  nOE = 0;
  nWE = 1;
  #100;
  for (i = 0; i < rdbytes; i = i + 1) begin
    fork
      A = ap;
      #(Th_oeh_weh) nCE = 0;
      #(Th_oeh_weh+Ta_nce) Save_Data(ap);
      #(Th_oeh_weh+Tcyr) nCE = 1;
    join 
    ap = ap + 1;  
  end   
  nOE = 1;
  #(Tpdz_oe_d) drive = 1'b1;
end
endtask


task CMD_Read_by_A;
input [A_MSB-1:0] addr;     // initial address to read
input [31:0]      rdbytes;  // number of bytes to read

reg   [A_MSB-1:0] ap;
integer i;
begin
  match = 1'b1;
  ap = addr;
  drive = 1'b0;
  nWE = 1;
  nCE = 0;
  nOE = 0;
  #10;
  for (i = 0; i < rdbytes; i = i+1) begin
    fork
      A = ap;
      #(Ta_a)   Save_Data(ap);
      #(Tcyr); 
    join  
    ap = ap + 1;  
  end   
  nCE = 1;
  nOE = 1; 
  #(Tpdz_oe_d) drive = 1'b1;    
end
endtask

task CMD_Program;
input [A_MSB-1:0] addr;
input [D_MSB-1:0] data;
integer i, p;
begin
  p = 2;

  `ifdef __VER
  match=1'b1;
  $display("RUN[Program]: addr=%h, word=%h", addr, data); 
  `endif

  Seq(12'h555, 8'hAA, 1, 0);
  Seq(12'h2AA, 8'h55, 0, 0);
  Seq(12'h555, 8'hA0, 0, 0);
  Event(addr, data, 0 , 1);

  #(Th_oeh_weh);

  CMD_Read_Status(addr);

  for(i=0; i<p; i=i+1) begin
    #(10*us_unit); 
    CMD_Read_Status(addr);
    if (DOUT_FLASH[7]==data[7]) begin 
      p=0;
      `ifdef __VER
      match=1'b1;
      $display("DONE[Program]"); 
      `endif
    end else if (!DOUT_FLASH[5]) begin 
      p=p+1;
    end else begin
      p=0;
      #100;
      CMD_Reset;
      `ifdef __VER
      match=1'b0;
      $display("ERROR[Program]: Operation Fail =("); 
      `endif
    end
  end

  `ifdef __VER
  mem[addr]<=data;
  `endif
  #100;
end
endtask


task CMD_Erase_Page;
input [A_MSB-A_SEC_MSB-1:0] sa; // sector address to erase
input [6:0] pa; // page address to erase
integer i, j, p;
begin
  p = 2;

  `ifdef __VER
  match=1'b1;
  $display("RUN[ErasePage], Sector %d Page %d", sa, pa); 
  `endif

  Seq(12'h555, 8'hAA, 1, 0);
  Seq(12'h2AA, 8'h55, 0, 0);
  Seq(12'h555, 8'h80, 0, 0);
  Seq(12'h555, 8'hAA, 0, 0);
  Seq(12'h2AA, 8'h55, 0, 0);
  Event({sa, pa, 3'h0, 8'h00}, 8'h50, 0, 1);  

  #Th_oeh_weh;

  CMD_Read_Status({sa, pa, 3'h0, 8'h00});

  for(i=0; i<p; i=i+1) begin
    #(50*ms_unit);
    CMD_Read_Status({sa, pa, 3'h0, 8'h00});  
    if (DOUT_FLASH[7]) begin 
      p=0;
      `ifdef __VER
      match=1'b1;
      $display("DONE[ErasePage]"); 
      `endif
    end else if (!DOUT_FLASH[5]) begin 
      p=p+1;
    end else begin
      p=0;
      #100;
      CMD_Reset;
      `ifdef __VER
      match=1'b0;
      $display("ERROR[ErasePage]: ErasePage Error =("); 
      `endif
    end
  end

  `ifdef __VER
  for (j=0; j<sizePage; j=j+1) mem[{sa, pa, 3'h0, 8'h00}+j]<={D_MSB{1'b1}};
  `endif
  #100;
end
endtask


task CMD_Erase_Sector;
input [A_MSB-A_SEC_MSB-1:0] addr0; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr1; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr2; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr3; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr4; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr5; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr6; // sector address to erase
input [A_MSB-A_SEC_MSB-1:0] addr7; // sector address to erase
input [3:0] n;
integer i, j, k, l, p;
reg [A_MSB-A_SEC_MSB-1:0] addr [7:0];
begin
  addr[0] = addr0;  
  addr[1] = addr1;
  addr[2] = addr2;
  addr[3] = addr3;
  addr[4] = addr4;
  addr[5] = addr5;
  addr[6] = addr6;
  addr[7] = addr7; 

  for(j=0; j<n; j=j+1) begin
    `ifdef __VER
      match=1'b1;
      $display("RUN[EraseSector], Sector %d",addr[j]); 
    `endif
    p = 2;

    Seq(12'h555, 8'hAA, 1, 0);
    Seq(12'h2AA, 8'h55, 0, 0);
    Seq(12'h555, 8'h80, 0, 0);
    Seq(12'h555, 8'hAA, 0, 0);
    Seq(12'h2AA, 8'h55, 0, 0);
    Event({addr[j], {A_SEC_MSB{1'b0}}}, 8'h30, 0, 1);  
    #(Tw_sec_load * us_unit);
  end  

  for(i=0; i<p; i=i+1) begin
    #(1000000);
    CMD_Read_Status({addr[0], {A_SEC_MSB{1'b0}}});  
    if (DOUT_FLASH[7]) begin 
      p=0;
      `ifdef __VER
      match=1'b1;
      $display("DONE[EraseSector]"); 
      `endif
    end else if (!DOUT_FLASH[5]) begin 
      p=p+1;
    end else begin
      p=0;
      #100;
      CMD_Reset;
      `ifdef __VER
      match=1'b0;
      $display("ERROR[EraseSector]: Operation Fail =("); 
      `endif
    end
  end

  `ifdef __VER
  for(k=0; k<n; k=k+1) begin
   for (l=0; l<sizeSector; l=l+1) mem[{addr[k],{A_SEC_MSB{1'b0}}}+l]<={D_MSB{1'b1}};
  end
  `endif
  #100;
end
endtask

task CMD_Erase_Chip;
integer i, j, k, l, p;
begin
  `ifdef __VER
  match=1'b1;
  $display("RUN[EraseChip]"); 
  `endif
  p = 2;

  Seq(11'h555, 8'hAA, 1, 0);
  Seq(11'h2AA, 8'h55, 0, 0);
  Seq(11'h555, 8'h80, 0, 0);
  Seq(11'h555, 8'hAA, 0, 0);
  Seq(11'h2AA, 8'h55, 0, 0);
  Seq(11'h555, 8'h10, 0, 1);   

  for(i=0; i<p; i=i+1) begin
    #(1000000);
    CMD_Read_Status(0);
    if (DOUT_FLASH[7]) begin 
      p=0;
      `ifdef __VER
      match=1'b1;
      $display("DONE[EraseChip]"); 
      `endif
    end else if (!DOUT_FLASH[5]) begin 
      p=p+1;
    end else begin 
      p=0;
      #100;
      CMD_Reset;
      `ifdef __VER
      match=1'b0;
      $display("ERROR[EraseChip]: EraseChip Error =("); 
      `endif
    end
  end
 
  `ifdef __VER
  for (i=0; i<sizeMem; i=i+1) mem[i]<={D_MSB{1'b1}};
  `endif
  #100;
end
endtask

task CMD_Read_Status;
input [A_MSB-1:0] addr; // initial address to read
begin
  drive = 1'b0;
  fork
    A = addr;
    nCE = 0;
    #(10) nWE = 1;
    #(10+Th_oeh_weh) nOE = 0;
    #(10+Th_oeh_weh+Ta_noe) DOUT_FLASH = D;
    #(10+Th_oeh_weh+Tcyr) nCE = 1;
    #(10+Th_oeh_weh+Tcyr) nOE = 1;
  join 

  #(Tpdz_oe_d) drive = 1'b1;
end
endtask

task CMD_Autoselect;
begin
  Seq(12'h555, 8'hAA, 1, 0);
  Seq(12'h2AA, 8'h55, 0, 0);
  Seq(12'h555, 8'h90, 0, 1);
 
  drive = 1'b0;
  nCE = 0;
  nOE = 0;

  fork
    A = {{(A_MSB-2){1'b0}}, 2'b00};
    #(Ta_a) DOUT_FLASH = D;
    #(Tcyr) A = {A_MSB{1'bx}}; 
  join 

  `ifdef __VER
  if (DOUT_FLASH[7:0]!=8'h01) begin
    match=1'b0;
    $display("ERROR[Autoselect CMD]: Manufactury ID: %h is not valid", DOUT_FLASH[7:0]);  
  end else begin
    match=1'b1;
    $display("EVENT[Autoselect CMD]: Manufactury ID: %h", DOUT_FLASH[7:0]); 
  end
  `endif 

  fork
    A = {{(A_MSB-2){1'b0}}, 2'b01};
    #(Ta_a) DOUT_FLASH = D;
    #(Tcyr) A = {A_MSB{1'bx}}; 
  join 

  `ifdef __VER
  if (DOUT_FLASH[7:0]!=8'hC8) begin
    match=1'b0;
    $display("ERROR[Autoselect CMD]: Device ID: %h is not valid", DOUT_FLASH[7:0]);  
  end else begin
    match=1'b1;
    $display("EVENT[Autoselect CMD]: Device ID: %h", DOUT_FLASH[7:0]); 
  end
  `endif 

  nCE = 1;
  nOE = 1;
  #(Tpdz_oe_d) drive = 1'b1;
end
endtask

task CMD_Autoselect_RdPrSec;
input [A_MSB-A_SEC_MSB-1:0] SA;
begin
  Seq(12'h555, 8'hAA, 1, 0);
  Seq(12'h2AA, 8'h55, 0, 0);
  Seq(12'h555, 8'h90, 0, 1);

  drive = 1'b0;
  nCE = 0;
  nOE = 0;
  fork
    A = {SA, {(A_SEC_MSB-2){1'b0}}, 2'b10};
    #(Ta_a) DOUT_FLASH = D;
    #(Tcyr) A = {A_MSB{1'bx}}; 
  join 

  `ifdef __VER
  if (DOUT_FLASH[0]) begin
    $display("EVENT[Autoselect CMD]: Sector %d Protect Status - Locked", SA); 
  end else begin
    $display("EVENT[Autoselect CMD]: Sector %d Protect Status - Unlocked", SA); 
  end
  `endif 

  nCE = 1;
  nOE = 1;
  #(Tpdz_oe_d) drive = 1'b1;
end
endtask

task CMD_Reset;
begin
  Event({A_MSB{1'bx}}, 8'hF0, 1, 1);

  `ifdef __VER
  $display("EVENT[Reset]: Applying Reset Command"); 
  `endif   

end
endtask

task CMD_Protect_Sector;
input [A_MSB-A_SEC_MSB-1:0] SA;
integer i;
time Tstart;
begin
  Tstart = 10 * us_unit;

  nCE = 1'b0;
  nOE = 1'b1;
  nWE = 1'b1;

  A[A_MSB-1:A_SEC_MSB] = {(A_MSB-A_SEC_MSB){1'bx}};
  A[6] = 1'bx;
  A9_HV = 1'b0;
  OE_HV = 1'b0;

  fork
    #(Tstart - (Tsu_a9_pr * us_unit)) A[A_MSB-1:A_SEC_MSB] = SA;
    #(Tstart - (Tsu_a9_pr * us_unit)) A[6] = 1'b0;
    #(Tstart - (Tsu_a9_pr * us_unit)) A9_HV = 1'b1;
    #(Tstart - (Tsu_oe_pr * us_unit)) OE_HV = 1'b1;

    #(Tstart) nWE = 1'b0;
    #(Tstart + (Tw_we_pr * us_unit)) nWE = 1'b1;
  join

  #(Tsu_oe_pr * us_unit) OE_HV = 1'b0;

  drive = 1'b0;

  fork
    A[1:0] = 2'b10;
    #(Th_oeh_weh)        nOE = 0;  
    #(Th_oeh_weh+Tcyr)   nOE = 1;
    #(Th_oeh_weh+Tcyr)   A   = {A_MSB{1'bx}};
    #(Th_oeh_weh+Ta_noe) DOUT_FLASH = D;
    #(Th_oeh_weh+Tcyr);
  join

  #(Tpdz_oe_d) drive = 1'b1;

  `ifdef __VER
  if (DOUT_FLASH[0]) begin
    $display("EVENT[Autoselect CMD]: Sector %d Protect Status - Locked", SA); 
  end else begin
    $display("EVENT[Autoselect CMD]: Sector %d Protect Status - Unlocked", SA); 
  end
  `endif 

  nCE = 1'b1;
  nOE = 1'b1;
  nWE = 1'b1;

  A9_HV = 1'b0;
  OE_HV = 1'b0;

  #(1000);
  CMD_Reset;
  nCE = 1;
  #(100);
end
endtask


task CMD_Unprotect_Sector;
input [A_MSB-A_SEC_MSB-1:0] SA;
integer i;
time Tstart;
begin
  Tstart = 10 * us_unit;

  nCE = 1'b0;
  nOE = 1'b1;
  nWE = 1'b1;

  A[A_MSB-1:A_SEC_MSB] = {(A_MSB-A_SEC_MSB){1'bx}};
  A[6] = 1'bx;
  A9_HV = 1'b0;
  OE_HV = 1'b0;

  fork
    #(Tstart - (Tsu_a9_pr * us_unit)) A[A_MSB-1:A_SEC_MSB] = SA;
    #(Tstart - (Tsu_a9_pr * us_unit)) A[6] = 1'b1;
    #(Tstart - (Tsu_a9_pr * us_unit)) A9_HV = 1'b1;
    #(Tstart - (Tsu_oe_pr * us_unit)) OE_HV = 1'b1;

    #(Tstart) nWE = 1'b0;
    #(Tstart + (Tw_pr_we * ms_unit)) nWE = 1'b1;
  join

  #(Tsu_oe_pr * us_unit) OE_HV = 1'b0; 

  drive = 1'b0;

  fork
    A[1:0] = 2'b10;
    #(Th_oeh_weh)        nOE = 0;  
    #(Th_oeh_weh+Tcyr)   nOE = 1;
    #(Th_oeh_weh+Tcyr)   A   = {A_MSB{1'bx}};
    #(Th_oeh_weh+Ta_noe) DOUT_FLASH = D;
    #(Th_oeh_weh+Tcyr);
  join

  #(Tpdz_oe_d) drive = 1'b1;

  `ifdef __VER
  if (DOUT_FLASH[0]) begin
    $display("EVENT[Autoselect]: Sector %d Protect Status - Locked", SA); 
  end else begin
    $display("EVENT[Autoselect]: Sector %d Protect Status - Unlocked", SA); 
  end
  `endif 

  nCE = 1'b1;
  nOE = 1'b1;
  nWE = 1'b1;

  A9_HV = 1'b0;
  OE_HV = 1'b0;

  #(1000);
  CMD_Reset;  
  nCE = 1;
  #(100);
end
endtask
//=============================================================================
//=============================================================================

endmodule